<?php
/*--------------------------------------------------------------------------------------------------
    GroupRepositoryFactory.php 2020-02-17
    Gambio GmbH
    http://www.gambio.de
    Copyright (c) 2020 Gambio GmbH
    Released under the GNU General Public License (Version 2)
    [http://www.gnu.org/licenses/gpl-2.0.html]
    --------------------------------------------------------------------------------------------------
 */
declare(strict_types=1);
namespace Gambio\Shop\ProductModifiers\Database;

use Gambio\Shop\ProductModifiers\Database\Core\Factories\Interfaces\GroupRepositoryFactoryInterface;
use Gambio\Shop\ProductModifiers\Database\Core\Factories\Interfaces\ImplementationFactoryInterface;
use Gambio\Shop\ProductModifiers\Database\Core\GroupRepository;
use Gambio\Shop\ProductModifiers\Database\Core\Mappers\Interfaces\GroupMapperInterface;
use Gambio\Shop\ProductModifiers\Database\Core\Readers\GroupReaderComposite;
use Gambio\Shop\ProductModifiers\Database\Core\Readers\Interfaces\GroupReaderCompositeInterface;
use Gambio\Shop\ProductModifiers\Database\Core\Readers\Interfaces\ModifierReaderCompositeInterface;
use Gambio\Shop\ProductModifiers\Database\Core\Readers\ModifierReaderComposite;
use Gambio\Shop\Attributes\ProductModifiers\Database\ImplementationFactory as AttributesImplementationFactory;
use Gambio\Shop\Properties\ProductModifiers\Database\ImplementationFactory as PropertyImplementationFactory;
use Gambio\Shop\ProductModifiers\Database\Presentation\PresentationMapperFactory;
use Gambio\Shop\ProductModifiers\Groups\Repositories\GroupRepositoryInterface;
use Gambio\Shop\ProductModifiers\Presentation\PresentationFactoryInterface;

/**
 * Class GroupRepositoryFactory
 * @package Gambio\Shop\ProductModifiers\Database
 */
class GroupRepositoryFactory implements GroupRepositoryFactoryInterface
{
    
    /**
     * @var ImplementationFactoryInterface[]
     */
    protected $implementations;

    
    
    /**
     * GroupRepositoryFactory constructor.
     *
     * @param ImplementationFactoryInterface[] $implementations
     */
    public function __construct(
        ImplementationFactoryInterface ...$implementations
    ) {
        $this->implementations    = count($implementations) ? $this->implementations :  $this->createImplementationList();
    }
    
    
    /**
     * @return ImplementationFactoryInterface[]
     * @codeCoverageIgnore
     */
    protected function createImplementationList(): array
    {
        $mapper = (new PresentationMapperFactory())->createMapperChain();
        return [
            new AttributesImplementationFactory($mapper),
            new PropertyImplementationFactory($mapper)
        ];
    }
    
    
    /**
     * @inheritDoc
     */
    public function createRepository(): GroupRepositoryInterface
    {
        return new GroupRepository($this->createGroupMapper(),
                                   $this->createGroupReaderComposite(),
                                   $this->createModifierReaderComposite());
    }
    
    
    /**
     * Create a Group Mappers Chain of Responsibility
     * @return GroupMapperInterface
     */
    protected function createGroupMapper(): GroupMapperInterface
    {
        $lastMapper = null;
        $mapper     = null;
        
        $groupMappers = [];
        foreach ($this->implementations as $factory) {
            if ($mapper) {
                $lastMapper = $lastMapper ?? $mapper;
                $newMapper  = $factory->createGroupMapper();
                /**
                 * @var GroupMapperInterface $lastMapper
                 */
                $lastMapper->setNext($newMapper);
                $lastMapper = $newMapper;
            } else {
                $mapper = $factory->createGroupMapper();
            }
        }
        
        return $mapper;
    }
    
    
    /**
     * @return GroupReaderCompositeInterface
     */
    protected function createGroupReaderComposite(): GroupReaderCompositeInterface
    {
        $groupCompositeReaders = [];
        foreach ($this->implementations as $factory) {
            $groupCompositeReaders[] = $factory->createGroupReader();
        }
        
        return new GroupReaderComposite(...$groupCompositeReaders);
    }
    
    
    /**
     * @return ModifierReaderCompositeInterface
     */
    protected function createModifierReaderComposite(): ModifierReaderCompositeInterface
    {
        $modifierCompositeReaders = [];
        foreach ($this->implementations as $factory) {
            $modifierCompositeReaders[] = $factory->createModifierReader();
        }
        
        return new ModifierReaderComposite(...$modifierCompositeReaders);
    }
    
    
}